// Serial Number Handling
document.addEventListener('DOMContentLoaded', function() {
    const generateSerialsBtn = document.getElementById('generateSerialsBtn');
    const manualSerialsBtn = document.getElementById('manualSerialsBtn');
    const serialInput = document.getElementById('adjust_serial');
    const quantityInput = document.getElementById('adjust_quantity');
    const serialError = document.getElementById('serialError');
    const manualAdjustmentForm = document.getElementById('manualAdjustmentForm');
    const manualSerialsModal = new bootstrap.Modal(document.getElementById('manualSerialsModal'));
    const serialNumberInputs = document.getElementById('serialNumberInputs');
    const manualQuantityDisplay = document.getElementById('manualQuantityDisplay');
    const saveManualSerials = document.getElementById('saveManualSerials');
    const manualSerialError = document.getElementById('manualSerialError');
    const productIdInput = document.getElementById('adjust_product_id');
    const submitButton = document.querySelector('button[type="submit"]');

    // Function to update validation status
    function updateValidationStatus(isValid, message = '') {
        if (serialError) {
            serialError.textContent = message;
            serialError.style.display = isValid ? 'none' : 'block';
            serialError.className = isValid ? 'text-success' : 'text-danger';
        }
        if (submitButton) {
            submitButton.disabled = !isValid;
        }
    }

    // Generate serial numbers with SN prefix
    if (generateSerialsBtn) {
        generateSerialsBtn.addEventListener('click', async function() {
            const quantity = parseInt(quantityInput.value);
            if (!quantity || quantity <= 0) {
                updateValidationStatus(false, 'Please enter a valid quantity first');
                return;
            }

            try {
                const response = await fetch('../api/handle_serial_numbers.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=generate&quantity=${quantity}&prefix=SN`
                });

                const data = await response.json();
                if (data.error) {
                    throw new Error(data.error);
                }

                serialInput.value = data.serials.join(', ');
                validateSerialNumbers();
            } catch (error) {
                updateValidationStatus(false, 'Error generating serial numbers: ' + error.message);
            }
        });
    }

    // Manual serial number entry
    if (manualSerialsBtn) {
        manualSerialsBtn.addEventListener('click', function() {
            const quantity = parseInt(quantityInput.value);
            if (!quantity || quantity <= 0) {
                updateValidationStatus(false, 'Please enter a valid quantity first');
                return;
            }

            // Update quantity display
            manualQuantityDisplay.textContent = quantity;

            // Clear previous inputs
            serialNumberInputs.innerHTML = '';

            // Create input fields
            for (let i = 0; i < quantity; i++) {
                const inputGroup = document.createElement('div');
                inputGroup.className = 'input-group mb-2';
                inputGroup.innerHTML = `
                    <span class="input-group-text">SN</span>
                    <input type="text" class="form-control serial-input" 
                           placeholder="Enter numbers only" required
                           oninput="this.value = this.value.replace(/[^0-9]/g, '')"
                           maxlength="5">
                `;
                serialNumberInputs.appendChild(inputGroup);
            }

            // Show modal
            manualSerialsModal.show();
        });
    }

    // Save manual serial numbers
    if (saveManualSerials) {
        saveManualSerials.addEventListener('click', async function() {
            const inputs = document.querySelectorAll('.serial-input');
            const serials = Array.from(inputs).map(input => {
                const value = input.value.trim();
                return value.startsWith('SN') ? value : `SN${value.padStart(5, '0')}`;
            }).filter(value => value);

            const quantity = parseInt(quantityInput.value);
            
            if (serials.length !== quantity) {
                updateValidationStatus(false, `Please enter exactly ${quantity} serial number(s)`);
                return;
            }

            // Check for duplicates
            if (new Set(serials).size !== serials.length) {
                updateValidationStatus(false, 'Duplicate serial numbers are not allowed');
                return;
            }

            try {
                const response = await fetch('../api/handle_serial_numbers.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=validate&serial_numbers=${JSON.stringify(serials)}&quantity=${quantity}`
                });

                const data = await response.json();
                if (data.error) {
                    throw new Error(data.error);
                }

                if (!data.valid) {
                    updateValidationStatus(false, data.message || 'Invalid serial numbers');
                    return;
                }

                // Update main input and close modal
                serialInput.value = serials.join(', ');
                updateValidationStatus(true, 'Serial numbers validated successfully');
                manualSerialsModal.hide();
            } catch (error) {
                updateValidationStatus(false, error.message);
            }
        });
    }

    // Validate serial numbers
    async function validateSerialNumbers() {
        if (!serialInput || !quantityInput) return;
        
        const serials = serialInput.value.split(/[\n,]/).map(s => s.trim()).filter(s => s);
        const quantity = parseInt(quantityInput.value);

        if (!serials.length || !quantity) {
            updateValidationStatus(false, 'Please enter both quantity and serial numbers');
            return;
        }

        if (serials.length !== quantity) {
            updateValidationStatus(false, `Number of serial numbers (${serials.length}) must match the quantity (${quantity})`);
            return;
        }

        try {
            const response = await fetch('../api/handle_serial_numbers.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=validate&serial_numbers=${JSON.stringify(serials)}&quantity=${quantity}`
            });

            const data = await response.json();
            if (data.error) {
                throw new Error(data.error);
            }

            updateValidationStatus(data.valid, data.message || (data.valid ? 'Serial numbers validated successfully' : 'Invalid serial numbers'));
        } catch (error) {
            updateValidationStatus(false, 'Error validating serial numbers: ' + error.message);
        }
    }

    // Handle form submission
    if (manualAdjustmentForm) {
        manualAdjustmentForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            const productId = document.getElementById('adjust_product_id').value;
            const quantity = parseInt(quantityInput.value);
            const serials = serialInput.value.split(/[\n,]/).map(s => s.trim()).filter(s => s);

            if (!productId || !quantity || !serials.length) {
                updateValidationStatus(false, 'Please fill in all required fields');
                return;
            }

            if (serials.length !== quantity) {
                updateValidationStatus(false, `Number of serial numbers (${serials.length}) must match the quantity (${quantity})`);
                return;
            }

            try {
                const response = await fetch('../api/handle_serial_numbers.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=save&product_id=${productId}&quantity=${quantity}&serial_numbers=${JSON.stringify(serials)}`
                });

                const data = await response.json();
                if (data.error) {
                    throw new Error(data.error);
                }

                // Show success message and close modal
                const successModal = new bootstrap.Modal(document.getElementById('successModal'));
                document.getElementById('successMessage').textContent = 'Stock adjusted successfully with serial numbers';
                successModal.show();

                // Reset form and close modal after 2 seconds
                setTimeout(() => {
                    manualAdjustmentForm.reset();
                    bootstrap.Modal.getInstance(document.getElementById('manualAdjustmentModal')).hide();
                    location.reload(); // Refresh to show updated stock
                }, 2000);

            } catch (error) {
                updateValidationStatus(false, 'Error saving serial numbers: ' + error.message);
            }
        });
    }

    // Validate serial numbers when quantity changes
    if (quantityInput) {
        quantityInput.addEventListener('input', validateSerialNumbers);
        quantityInput.addEventListener('change', validateSerialNumbers);
    }
    if (serialInput) {
        serialInput.addEventListener('input', validateSerialNumbers);
    }
}); 